// pages/api/admin/users/index.ts
import type { NextApiRequest, NextApiResponse } from "next";
import { prisma } from "@/lib/prisma";
import { requireAdmin } from "@/lib/admin";

export default async function handler(
  req: NextApiRequest,
  res: NextApiResponse
) {
  const session = await requireAdmin(req, res);
  if (!session) return;

  if (req.method !== "GET")
    return res.status(405).json({ ok: false, error: "Method not allowed" });

  // Parse query parameters for filters
  const { role: roleFilter, search } = req.query;

  // Get all users first to filter by role (case-insensitive)
  let where: any = {};

  // Search filter
  if (search) {
    where.OR = [
      { name: { contains: search as string, mode: "insensitive" } },
      { email: { contains: search as string, mode: "insensitive" } },
    ];
  }

  const allUsers = await prisma.user.findMany({
    where,
    orderBy: { createdAt: "desc" },
    select: {
      id: true,
      email: true,
      name: true,
      role: true,
      ethAddress: true,
      hourlyRate: true,
      isExternal: true,
      companyName: true,
      createdAt: true,
      customRole: {
        select: {
          id: true,
          name: true,
          description: true,
        },
      },
    },
  });

  // Filter by user role if specified (case-insensitive)
  let users = allUsers;
  if (roleFilter) {
    users = allUsers.filter(
      (user) => user.role?.toUpperCase() === String(roleFilter).toUpperCase()
    );
  }

  res.status(200).json({ ok: true, users });
}
